/*****************************************************************************
 * Project: SCT start stop Example for LPC54xx
 *
 * Implements a start stop event example for the SCT
 * For this application the following interface pins are used:
 * -------------------------------------------------------------------------
 * Signal      Port               Signal
 * -------------------------------------------------------------------------
 * SCT_OUT0   P0_7         		COUNTER_L_RUN
 * SCT_OUT1   P0_8        		COUNTER_H_RUN
 * SCT_OUT2   P0_9         		STATE_1
 * SCT_OUT3   P0_10        		STATE_2
 * ------------------------------------------------------------------------*/

#include "board.h"

#define counterH_start (5000)
#define counterH_stop  (2000)

#define counterL_start (3000)
#define counterL_stop  (2500)

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
void start_stop_init(void);
/*****************************************************************************
 * Public functions
 ****************************************************************************/
void start_stop_init (void)
{
	Chip_SCT_Init(LPC_SCT);                			   								// enable the SCT clock


	Chip_SCT_SetMatchCountL(LPC_SCT, SCT_MATCH_0, counterL_stop);					// deactivate_L
	Chip_SCT_SetMatchReloadL(LPC_SCT, SCT_MATCH_0, counterL_stop);

	Chip_SCT_SetMatchCountH(LPC_SCT, SCT_MATCH_0, counterL_start);					// activate_L
	Chip_SCT_SetMatchReloadH(LPC_SCT, SCT_MATCH_0, counterL_start);

	Chip_SCT_SetMatchCountL(LPC_SCT, SCT_MATCH_1, counterH_start);					// activate_H
	Chip_SCT_SetMatchReloadL(LPC_SCT, SCT_MATCH_1, counterH_start);

	Chip_SCT_SetMatchCountH(LPC_SCT, SCT_MATCH_1, counterH_stop);					// deactivate_H
	Chip_SCT_SetMatchReloadH(LPC_SCT, SCT_MATCH_1, counterH_stop);


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0 , ENABLE_STATE1);									// event 0 happens in state 1
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	  |	// MATCHSEL [3:0]   = related to match 0
                                                                        SCT_HEVENT_L		  |	// HEVENT   [4]     = use L state & match
                                                                        SCT_COMBMODE_MATCH	  |	// COMBMODE [13:12] = match AND IO condition
                                                                        SCT_STATELD_1		  |	// STATELD  [14]    = STATEV is loaded into state
                                                                        SCT_STATEEV_0		 ));// STATEV   [19:15] = new state is 0


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1 , ENABLE_STATE0);										// event 1 happens in state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1,  (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match 0
                                                                         SCT_HEVENT_H		    |	// HEVENT   [4]     = use H state & match
                                                                         SCT_COMBMODE_MATCH	  	|	// COMBMODE [13:12] = match AND IO condition
                                                                         SCT_STATELD_1		    |	// STATELD  [14]    = STATEV is loaded into state
                                                                         SCT_STATEEV_1		  ));	// STATEV   [19:15] = new state is 1


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_2 , ENABLE_STATE0);										              // event 2 happens in state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1		|	// MATCHSEL [3:0]   = related to match 1
                                                                        SCT_HEVENT_L		    |	// HEVENT   [4]     = use L state & match
                                                                        SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match AND IO condition
                                                                        SCT_STATELD_1		    |	// STATELD  [14]    = STATEV is loaded into state
                                                                        SCT_STATEEV_1		    ));	// STATEV   [19:15] = new state is 1

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_3 , ENABLE_STATE1);										// event 3 happens in state 1
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_3,  (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1	|	// MATCHSEL [3:0]   = related to match 1
                                                                         SCT_HEVENT_H		    |	// HEVENT   [4]     = use H state & match
                                                                         SCT_COMBMODE_MATCH		|	// COMBMODE [13:12] = match AND IO condition
                                                                         SCT_STATELD_1		  	|	// STATELD  [14]    = STATEV is loaded into state
                                                                         SCT_STATEEV_0		  ));	// STATEV   [19:15] = new state is 0

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_0,  (CHIP_SCT_EVENT_T) (  SCT_EVT_1 |
                                                                    SCT_EVT_3 ));				// event 1 and 3 set OUTP_0   -> counter L RUN

	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_0, (CHIP_SCT_EVENT_T) ( SCT_EVT_0 |
                                                                    SCT_EVT_2 ));				// event 0 and 2 clear OUTP_0

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T) (   SCT_EVT_0 |
                                                                    SCT_EVT_2 ));				// event 0 and 2 set OUTP_1   -> counter H RUN

	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T) ( SCT_EVT_1 |
                                                                    SCT_EVT_3 ));				// event 1 and 3 clear OUTP_1

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_2, SCT_EVT_2 );										// event 2 will set OUTP_2    -> STATE 1
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_2, SCT_EVT_0);										// event 0 will clear OUTP_2

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_3, SCT_EVT_1);										  // event 1 will set OUTP_3    -> STATE 2
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_3, SCT_EVT_3);										// event 3 will clear OUTP_3


	Chip_SCT_Output(LPC_SCT,0x01);           							          // OUTP_0 high OUTP_1/2/3 low

    LPC_SCT->STOP_L     = SCT_EVT_0 | SCT_EVT_2;            			// events 0 and 2 STOP the L counter
    LPC_SCT->START_L    = SCT_EVT_1 | SCT_EVT_3;            			// events 1 and 3 START the L counter
    LPC_SCT->LIMIT_L    = SCT_EVT_0 | SCT_EVT_2;            			// events 0 and 2 as L counter limit

    LPC_SCT->STOP_H     = SCT_EVT_1 | SCT_EVT_3;            			// events 1 and 3 STOP the H counter
    LPC_SCT->START_H    = SCT_EVT_0 | SCT_EVT_2;            			// events 0 and 2 START the H counter
    LPC_SCT->LIMIT_H    = SCT_EVT_1 | SCT_EVT_3;            			// events 1 and 3 as H counter limit

    LPC_SCT->CTRL_U		=			(0  << 0)       |            		      // DOWN_L [0]     = L counting down
                            		(0  << 1)       |           		  // STOP_L [1]     = L counter does run
                            		(0  << 2)       |            		  // HALT_L [2]     = unhalt the L counter
                            	SCT_CTRL_CLRCTR_L  	|            		// CLRT_L [3]     = clear L counter
                            	SCT_CTRL_BIDIR_L(0) |            		// BIDIR_L[4]     = not bidirectional counting
                            	SCT_CTRL_PRE_L(11)  |            		// PRE_L  [12:5]  = L prescaler = 12
                            		(0  << 16)      |            		  // DOWN_H [16]    = H counting down
                            	SCT_CTRL_STOP_H 	|            		  // STOP_H [1]     = stop the H counter
                            	(0  << 18) 			|            		    // HALT_H [2]     = unhalt the H counter
                            	SCT_CTRL_CLRCTR_H 	|            		// CLRT_H [3]     = clear H counter
                            	SCT_CTRL_BIDIR_H(0) |            		// BIDIR_H[4]     = not bidirectional counting
                            	SCT_CTRL_PRE_H(11)  ;           		// PRE_H  [28:21] = H prescaler = 12

}


int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();

  LPC_SYSCON->AHBCLKDIV = 8;


	Chip_Clock_EnablePeriphClock(	(CHIP_SYSCON_CLOCK_T) ( SYSCON_CLOCK_GPIO0 	|			// enable GPIO port 2 clock
	                                                        SYSCON_CLOCK_IOCON	));		    // enable IOCON clock

	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 7, IOCON_FUNC2);

	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 8, IOCON_FUNC2);

	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 9, IOCON_FUNC2);

	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 10, IOCON_FUNC2);


	start_stop_init();


    while (1)                                              					// loop forever
    {
    	__WFI();
    }
}


